unit UfrmMain2;

interface

{.$I ..\..\Physics2D\Physics2D.inc}

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, ExtCtrls, Math,
  System.Generics.Collections,
  UOpenGLCanvas, UPhysics2D, UPhysics2DTypes, UPhysics2DHelper,
  UPingPongTypes, UGame;

type
  TfrmMain2 = class(TForm)
    imgDisplay: TImage;
    btnNewGame: TButton;
    clrBlockColor: TColorBox;
    cboBlockType: TComboBox;
    Bevel1: TBevel;
    btnPauseResume: TButton;
    btnLoadMap: TButton;
    btnSaveMap: TButton;
    chkEditMode: TCheckBox;
    OpenDialog: TOpenDialog;
    SaveDialog: TSaveDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure chkEditModeClick(Sender: TObject);
    procedure btnNewGameClick(Sender: TObject);
    procedure btnPauseResumeClick(Sender: TObject);
    procedure imgDisplayMouseDown(Sender: TObject; Button: TMouseButton;
       Shift: TShiftState; AX, AY: Integer);
    procedure clrBlockColorChange(Sender: TObject);
    procedure cboBlockTypeChange(Sender: TObject);
    procedure btnSaveMapClick(Sender: TObject);
    procedure btnLoadMapClick(Sender: TObject);
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
  private
      { Private declarations }
    GLCanvas: TGLCanvas;

    procedure Display;
    procedure SetEditingMap(Editing: Boolean);
    procedure ShowSelectedBlockProperty;
    procedure DoNewGame;
    procedure TimerProgress(const deltaTime, newTime: Double);
  public
      { Public declarations }
  end;

var
  frmMain2: TfrmMain2;

implementation

uses
   MSTimer, System.UITypes;

{$R *.dfm}

const
  PlateColor = TColors.Maroon;
  BallColor = TColors.Red;

{ TfrmMain }

procedure TfrmMain2.FormDestroy(Sender: TObject);
begin
  GameFree;

  MSCadencer.Free;
  GLCanvas.Free;
end;

procedure TfrmMain2.FormCreate(Sender: TObject);
begin
  GameInit(  70, //DefaultPlateWidth
  15, //DefaultPlateHeight
  4, //DefaultBallRadius
  600, //MapWidth
  500, //MapHeight
  40, //BlockWidth
  20, //BlockHeight
  50 //BlockMinTop
  );
  
  MSCadencer := TMSTimer.Create;
  MSCadencer.OnProgress := TimerProgress;

  GLCanvas := TGLCanvas.Create(imgDisplay, True, True, False, True);
  GLCanvas.DefaultFont.WinColor := clBlack;

  OpenDialog.InitialDir := ExtractFileDir(Application.ExeName);
  SaveDialog.InitialDir := OpenDialog.InitialDir;

  SetEditingMap(False);
end;

procedure TfrmMain2.SetEditingMap(Editing: Boolean);
var i: Integer; b: TBlock;
begin
  EditingMap := Editing;
  SelectedBlock := nil;

  if chkEditMode.Checked <> Editing then
    chkEditMode.Checked := Editing;

  clrBlockColor.Enabled := Editing;
  cboBlockType.Enabled := Editing;
  btnSaveMap.Enabled := Editing;
  btnPauseResume.Enabled := not Editing;
  btnNewGame.Enabled := not Editing;

  if Editing then
  begin
    MSCadencer.Enabled := False;

    FillChar(Map, SizeOf(Map), 0);

    for i := 0 to Blocks.Count - 1 do
    begin
      b := TBlock(Blocks[i]);
      b.RestoreHP;
      SetMap(b.X, b.Y, b);
    end;

    Display;
  end

  else
    DoNewGame;
end;

procedure TfrmMain2.DoNewGame;
var
   i, t: Integer;
   ax, ay: Integer;
   ABlock: TBlock;
begin
   GameState := gsNotLaunched;
   Life := 3;
   Plate.Width := DefaultPlateWidth;
   Plate.X := MapWidth div 2;
   Ball.X := Plate.X;
   Ball.Y := DefaultBallRadius + DefaultPlateHeight;
   btnPauseResume.Caption := 'Pause';

   if MapFileName = '' then
   begin
      if Assigned(b2World) then
         FreeAndNil(b2World);
      FreeAllBlocks;
      ay := 280;
      for i := 0 to 4 do
      begin
         ax := (MapWidth - BlockWidth * 9) div 2;
         for t := 0 to 8 do
         begin
            ABlock := TBlock.Create;
            ABlock.X := ax;
            ABlock.Y := ay;
            ABlock.Color := clBlue;
            ABlock.BlockType := btMud;
            Blocks.Add(ABlock);
            ax := ax + BlockWidth;
         end;
         ay := ay + BlockHeight;
      end;
      MapFileName := 'default';
   end;

   LeftBlockCount := Blocks.Count;
   for i := 0 to Blocks.Count - 1 do
      if TBlock(Blocks[i]).BlockType = btUnbreakable then
         Dec(LeftBlockCount);

   InitializePhysics;
   Display;

   MSCadencer.Enabled := True;
end;

procedure TfrmMain2.btnNewGameClick(Sender: TObject);
begin
  DoNewGame;
end;

procedure TfrmMain2.btnLoadMapClick(Sender: TObject);
var
  i, blockCount: Integer;
  stream: TMemoryStream;
  block: TBlock;
begin
  if OpenDialog.Execute then
  begin
    FreeAllBlocks;
    MapFileName := OpenDialog.FileName;

    stream := TMemoryStream.Create;
    try
      try
        stream.LoadFromFile(MapFileName);
        stream.Read(blockCount, SizeOf(blockCount));
        for i := 0 to blockCount - 1 do
        begin
          block := TBlock.Create;

          stream.Read(block.X, SizeOf(block.X));
          stream.Read(block.Y, SizeOf(block.Y));
          stream.Read(block.Color, SizeOf(block.Color));
          stream.Read(block.BlockType, SizeOf(block.BlockType));
          Blocks.Add(block);
        end;

        if EditingMap then
          SetEditingMap(True)
        else
          DoNewGame;

      except
        ShowMessage('Corrupted map file.');
        FreeAllBlocks;
        if EditingMap then
          SetEditingMap(True)
        else
          DoNewGame;
      end;

    finally
      stream.Free;
    end;
  end;
end;

procedure TfrmMain2.btnPauseResumeClick(Sender: TObject);
begin
   if EditingMap or (GameState in [gsNotLaunched, gsGameOver]) then
      Exit;
   if GameState = gsPaused then
   begin
      GameState := gsPlaying;
      btnPauseResume.Caption := 'Pause';
   end
   else
   begin
      GameState := gsPaused;
      btnPauseResume.Caption := 'Resume';
   end;

  MSCadencer.Enabled := GameState = gsPlaying;
end;

procedure TfrmMain2.btnSaveMapClick(Sender: TObject);
var
   i: Integer;
   fn: string;
   stream: TMemoryStream;
   block: TBlock;
begin
  if SaveDialog.Execute then
  begin
    fn := SaveDialog.FileName;
    stream := TMemoryStream.Create;
    try
      i := Blocks.Count;
      stream.Write(i, SizeOf(i));
      for i := 0 to Blocks.Count - 1 do
      begin
        block := TBlock(Blocks[i]);
        stream.Write(block.X, SizeOf(block.X));
        stream.Write(block.Y, SizeOf(block.Y));
        stream.Write(block.Color, SizeOf(block.Color));
        stream.Write(block.BlockType, SizeOf(block.BlockType));
      end;
      stream.SaveToFile(fn);
    finally
      stream.Free;
    end;
  end;
end;

procedure TfrmMain2.cboBlockTypeChange(Sender: TObject);
begin
  if EditingMap and Assigned(SelectedBlock) then
    SelectedBlock.BlockType := TBlockType(cboBlockType.ItemIndex);
end;

procedure TfrmMain2.chkEditModeClick(Sender: TObject);
begin
  SetEditingMap(chkEditMode.Checked);
end;

procedure TfrmMain2.clrBlockColorChange(Sender: TObject);
begin
  if EditingMap and Assigned(SelectedBlock) then
  begin
    SelectedBlock.Color := clrBlockColor.Selected;
    Display;
  end;
end;

procedure TfrmMain2.FormKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if EditingMap and Assigned(SelectedBlock) and (Key = 46) and
    (MessageDlg('Really want to delete selected block?', mtConfirmation, [mbYes, mbNo], 0) = mrYes) then  // delete
  begin
    SetMap(SelectedBlock.X, SelectedBlock.Y, nil);
    Blocks.Remove(SelectedBlock);
    FreeAndNil(SelectedBlock);
    ShowSelectedBlockProperty;
    Display;
  end;
end;

function GetReverseColor(Color: TColor): TColor;
begin
  Result := RGB(255-GetRValue(Color), 255-GetGValue(Color), 255-GetBValue(Color));
end;

procedure TfrmMain2.Display;
var
  i: Integer;
  tmpX: Single;
  b: TBlock;
begin
  GLCanvas.RenderingBegin(clWhite);

  for i := 0 to Blocks.Count-1 do
  begin
    b := TBlock(Blocks[i]);

    if b.HP <= 0 then
      Continue;

    GLCanvas.SetBrushColorWin(b.Color, 255, False).
      FillRect(b.X + 1, b.Y + 1, b.X + BlockWidth - 1, b.Y + BlockHeight - 1);

    if SelectedBlock = b then
    begin
      GLCanvas.SetPenWidth(1).SetPenColorWin(GetReverseColor(b.Color), 255,False).
        Line(b.X + 1, b.Y + BlockHeight - 1, b.X + BlockWidth - 1, b.Y + 1).
        Line(b.X + 1, b.Y + 1, b.X + BlockWidth - 1, b.Y + BlockHeight - 1);
    end;
  end;

  if not EditingMap then
  begin

    GLCanvas.TranslateX := Plate.X;
    GLCanvas.SetPenWidth(3).SetPenColorWin(PlateColor, 255, False).
      Polyline(TGLPointsF(Plate.DrawPoints), Plate.DrawPointCount);
    GLCanvas.TranslateX := 0;

    GLCanvas.SetBrushColorWin(BallColor, 255, False).
      FillEllipse(Ball.X, Ball.Y, DefaultBallRadius, DefaultBallRadius);

    tmpX := MapWidth - 7;
    for i := 1 to Life - 1 do
    begin
      GLCanvas.FillEllipse(tmpX, MapHeight - DefaultBallRadius - 2,
        DefaultBallRadius, DefaultBallRadius);
      tmpX := tmpX - DefaultBallRadius * 2 - 3;
    end;

    if GameState = gsNotLaunched then
      GLCanvas.TextOut('Launch the ball by left clicking.', 2, MapHeight - 11)
    else if GameState = gsGameOver then
      GLCanvas.TextOut('Game Over', 2, MapHeight - 11)
    else if GameState = gsGameFinished then
      GLCanvas.TextOut('Game Success', 2, MapHeight - 11);

  end;

  GLCanvas.RenderingEnd;
end;

procedure TfrmMain2.imgDisplayMouseDown(Sender: TObject; Button: TMouseButton;
   Shift: TShiftState; AX, AY: Integer);
var
  i: Integer;
begin
  if EditingMap then
  begin
    AY := MapHeight - AY;
    SelectedBlock := Map[AX div BlockHeight, AY div BlockHeight];

    if not Assigned(SelectedBlock) then
    begin

      if AY < BlockMinTop then
        ShowMessage('The brick cannot be too low.')
      else if (AX >= MapWidth - BlockWidth / 2) or Assigned(Map[AX div BlockHeight + 1][AY div BlockHeight]) then
        ShowMessage('Cannot add a block here.')
      else
      begin
        SelectedBlock := TBlock.Create;
        with SelectedBlock do
        begin
          X := (AX div BlockHeight) * BlockHeight;
          Y := (AY div BlockHeight) * BlockHeight;
          Color := clBlue;
          BlockType := btMud;
          RestoreHP;
        end;
        Blocks.Add(SelectedBlock);
        SetMap(AX, AY, SelectedBlock);
      end;
    end;

    ShowSelectedBlockProperty;
    Display;
  end

  else
  begin
    if GameState = gsNotLaunched then
    begin
      b2BallBody.SetLinearVelocity(BallInitialVelocity);
      GameState := gsPlaying;

      MSCadencer.Enabled := True;
    end;
  end;
end;

procedure TfrmMain2.ShowSelectedBlockProperty;
begin
   if Assigned(SelectedBlock) then
   begin
      cboBlockType.ItemIndex := Ord(SelectedBlock.BlockType);
      clrBlockColor.Selected := SelectedBlock.Color;
   end
   else
   begin
      cboBlockType.ItemIndex := -1;
      clrBlockColor.Selected := clNone;
   end;
end;

procedure TfrmMain2.TimerProgress(const deltaTime, newTime: Double);
var
   sp, cp: TPoint;
begin
   GetCursorPos(sp);
   cp := imgDisplay.ScreenToClient(sp);

   DoGameProgress(deltaTime,cp,imgDisplay.Width);

   Display;
end;

end.

